<?php

namespace App\Events;

use App\Models\Chat;
use App\Models\Message;
use Illuminate\Broadcasting\InteractsWithSockets;
use Illuminate\Broadcasting\PrivateChannel;
use Illuminate\Contracts\Broadcasting\ShouldBroadcast;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Contracts\Queue\ShouldQueueAfterCommit;
use Illuminate\Foundation\Events\Dispatchable;
use Illuminate\Queue\SerializesModels;

final class NewChatMessage implements ShouldBroadcast, ShouldQueue, ShouldQueueAfterCommit
{
    use Dispatchable, InteractsWithSockets, SerializesModels;

    public function __construct(
        public readonly Chat    $chat,
        public readonly Message $message,
    )
    {
        if (!$this->message->relationLoaded('sender')) {
            $this->message->load('sender');
        }
    }

    public function broadcastOn(): array
    {
        $channels = [
            new PrivateChannel('Support.Chat.' . $this->chat->id)
        ];

        if ($this->chat->type === 'individual') {
            $branchId = optional($this->chat->user)->branch_id;
        } else {
            $branchId = $this->chat->branch_id;
        }

        if ($branchId) {
            $channels[] = new PrivateChannel("Support.Chats.{$branchId}");
        }

        $channels[] = new PrivateChannel('Support.Chats');

        return $channels;
    }

    public function broadcastAs(): string
    {
        return 'NewChatMessage';
    }

    public function broadcastWith(): array
    {
        $this->chat->loadMissing(['user', 'lastMessage', 'branch', 'user.dietPlan', 'user.lastResult', 'user.lastVisit', 'user.subscription']);

        $chatData = [
            'id' => $this->chat->id,
            'status' => $this->chat->status,
            'type' => $this->chat->type,
            'archived' => false,
            'new' => true,
            'created_at' => $this->chat->created_at,
        ];

        if ($this->chat->lastMessage) {
            $chatData['last_message'] = [
                'id' => $this->chat->lastMessage->id,
                'text' => $this->chat->lastMessage->text,
            ];
            $chatData['last_message_created_at'] = $this->chat->lastMessage->created_at;
        }

        if ($this->chat->type === 'individual' && $this->chat->user) {
            $chatData['user'] = [
                'name' => $this->chat->user->name,
                'email' => $this->chat->user->email,
                'photo_url' => $this->chat->user->photo_url,
                'has_subscription' => $this->chat->user->has_subscription,
                'current_weight' => $this->chat->user->current_weight,
                'current_fats' => $this->chat->user->current_fats,
                'subscription' => $this->chat->user->subscription,
                'last_result' => $this->chat->user->lastResult,
            ];
        }

        if ($this->chat->type === 'group' && $this->chat->branch) {
            $chatData['branch'] = [
                'id' => $this->chat->branch->id,
                'name' => $this->chat->branch->name,
            ];
        }

        return [
            'chat_id' => $this->chat->id,
            'chat' => $chatData,
            'message' => $this->message,
        ];
    }
}
