<?php

namespace App\Http\Controllers;

use App\Actions\CreateSupportCallAction;
use App\Actions\GenerateLiveKitTokenAction;
use App\Enums\SupportCallStatus;
use App\Events\NewSupportCallEvent;
use App\Events\UpdateSupportCallEvent;
use App\Exceptions\LiveKitException;
use App\Http\Requests\CreateSupportCallRequest;
use App\Jobs\MarkCallAsMissedJob;
use App\Models\Chat;
use App\Models\SupportCall;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Http\Request;
use Inertia\Inertia;
use Random\RandomException;

class CallController extends Controller
{
    public function index(Request $request)
    {
        $q = $request->input('q', '');

        $calls = SupportCall::with(['user', 'chat', 'chat.user'])->latest()->get();

        return Inertia::render('Calls/Index', compact('calls', 'q'));
    }

    /**
     * @throws LiveKitException
     */
    public function show(Request $request, SupportCall $call, GenerateLiveKitTokenAction $tokenAction)
    {
        if ($call->status === SupportCallStatus::ENDED) {
            return to_route('chat.index')
                ->with('error', __('This call has ended.'));
        }

        $call->loadMissing('user');

        $q = $request->input('q', '');

        $chats = Chat::with([
            'user',
            'user.dietPlan',
            'user.dietPlan.category',
            'user.lastVisit',
            'user.lastResult',
            'user.subscription',
//            'messages',
//            'messages.media',
//            'messages.sender:id,name',
            'lastMessage',
        ])->when($q, function ($query) use ($q) {
            $query
                ->whereHas('user', function (Builder $query) use ($q) {
                    $query->where('name', 'like', '%' . $q . '%');
                })
                ->orWhereHas('messages', function (Builder $query) use ($q) {
                    $query->whereRaw("text COLLATE utf8mb4_general_ci LIKE ?", ['%' . $q . '%']);
                });
        })
            ->withAggregate('lastMessage', 'created_at')
            ->orderByDesc('last_message_created_at')
            ->limit(1)
            ->get();

        $calls = SupportCall::with('user')->where('status', SupportCallStatus::PENDING)
            ->get();

        $liveKitData = $tokenAction->handle($call, $request->user());

        $call->update([
            'status' => SupportCallStatus::IN_PROGRESS,
        ]);

        return Inertia::render('Calls/Show', compact('chats', 'calls', 'call', 'liveKitData', 'q'));
    }

    /**
     * @throws RandomException
     * @throws LiveKitException
     */
    public function store(
        CreateSupportCallRequest   $request,
        CreateSupportCallAction    $action,
        GenerateLiveKitTokenAction $tokenAction,
    )
    {
        $chat = Chat::with('user')->findOrFail($request->chat_id);
        $lastCall = $chat->calls()->latest()->first();
        // update chat admin_id if not equal it
        if ($lastCall && $lastCall->admin_id !== $request->user()->id) {
            $chat->update(['admin_id' => $request->user()->id]);
        }


        if ($lastCall && in_array($lastCall->status, [SupportCallStatus::PENDING, SupportCallStatus::IN_PROGRESS])) {
            return to_route('calls.show', $lastCall);
        }

        $call = $action->handle($request->validated(), $chat->user);
        // update chat admin_id if not equal it
        if ($chat->admin_id !== $request->user()->id) {
            $chat->update(['admin_id' => $request->user()->id]);
        }

        event(new NewSupportCallEvent($call));

        MarkCallAsMissedJob::dispatch($call)->delay(now()->addMinutes(5));

        return to_route('calls.show', $call);
    }

    public function destroy(SupportCall $call)
    {
        $call->update([
            'status' => SupportCallStatus::ENDED,
            'ended_at' => now(),
        ]);

        event(new UpdateSupportCallEvent($call));

        return to_route('chat.index');
    }
}
