<?php

namespace App\Http\Controllers;

use App\Actions\CreateChatAction;
use App\Enums\SupportCallStatus;
use App\Models\Branch;
use App\Models\Chat;
use App\Models\SupportCall;
use App\Models\User;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Http\Request;
use Inertia\Inertia;

class ChatController extends Controller
{
    public function index(Request $request)
    {
        $q = $request->input('q', '');
        $branchId = $request->input('branch_id');

        $user = auth()->user();
        $isSuperAdmin = $user->hasRole('super_admin');

        if ($isSuperAdmin) {
            $branches = Branch::select('id', 'name')->get();
            $branchIds = $branches->pluck('id')->toArray();
        } else {
            $branches = collect();
            $branchIds = $user->branches()->pluck('id')->toArray();
        }

        // Get group chats for branches
        $groupChats = Chat::group()
            ->when(! $isSuperAdmin, function ($query) use ($branchIds) {
                $query->whereIn('branch_id', $branchIds);
            })
            ->when($branchId, function ($query) use ($branchId) {
                $query->where('branch_id', $branchId);
            })
            ->when($q, function ($query) use ($q) {
                $query->where(function ($query) use ($q) {
                    $query->whereHas('branch', function (Builder $query) use ($q) {
                        $query->where('name', 'like', '%'.$q.'%');
                    })->orWhereHas('messages', function (Builder $query) use ($q) {
                        $query->whereRaw('text COLLATE utf8mb4_general_ci LIKE ?', ['%'.$q.'%']);
                    });
                });
            })
            ->with([
                'branch',
                'lastMessage',
                'lastMessage.media',
                'messages' => function ($query) use ($isSuperAdmin) {
                    if ($isSuperAdmin) {
                        $query->withTrashed();
                    }
                },
                'messages.media',
                'messages.sender',
                'messages.reactions',
                'messages.reactions.actor:id,name',
                'messages.replyTo' => function ($query) use ($isSuperAdmin) {
                    if ($isSuperAdmin) {
                        $query->withTrashed();
                    }
                },
                'messages.replyTo.sender',
                'messages.replyTo.media',
            ])
            ->withAggregate('lastMessage', 'created_at')
            ->withAggregate('lastMessage as last_message_sender_type', 'sender_type')
            ->orderByRaw('(last_message_sender_type = ?) DESC, last_message_created_at DESC', [User::class])
            ->get();

        // Get individual chats
        $individualChats = Chat::individual()
            ->when(! $isSuperAdmin, function ($query) use ($branchIds) {
                $query->whereHas('user', function (Builder $query) use ($branchIds) {
                    $query->whereIn('branch_id', $branchIds);
                });
            })
            ->when($branchId, function ($query) use ($branchId) {
                $query->whereHas('user', function (Builder $query) use ($branchId) {
                    $query->where('branch_id', $branchId);
                });
            })
            ->when($q, function ($query) use ($q) {
                $query->where(function ($query) use ($q) {
                    $query->whereHas('user', function (Builder $query) use ($q) {
                        $query->where('name', 'like', '%'.$q.'%');
                    })->orWhereHas('messages', function (Builder $query) use ($q) {
                        $query->whereRaw('text COLLATE utf8mb4_general_ci LIKE ?', ['%'.$q.'%']);
                    });
                });
            })
            ->withAggregate('lastMessage', 'created_at')
            ->withAggregate('lastMessage as last_message_sender_type', 'sender_type')
            ->orderByRaw('(last_message_sender_type = ?) DESC, last_message_created_at DESC', [User::class])
            ->with([
                'user',
                'user.dietPlan',
                'user.dietPlan.category',
                'user.lastVisit',
                'user.lastResult',
                'user.subscription',
                'lastMessage',
                'lastMessage.media',
                'messages' => function ($query) use ($isSuperAdmin) {
                    if ($isSuperAdmin) {
                        $query->withTrashed();
                    }
                },
                'messages.media',
                'messages.sender',
                'messages.reactions',
                'messages.reactions.actor:id,name',
                'messages.replyTo' => function ($query) use ($isSuperAdmin) {
                    if ($isSuperAdmin) {
                        $query->withTrashed();
                    }
                },
                'messages.replyTo.sender',
                'messages.replyTo.media',
            ])
            ->get();

        // Merge group chats at the top, then individual chats
        $chats = $groupChats->concat($individualChats);

        $calls = SupportCall::with('user')
            ->where('status', SupportCallStatus::PENDING)
            ->whereHas('user', function (Builder $query) use ($branchIds, $branchId) {
                if ($branchId) {
                    $query->where('branch_id', $branchId);
                } else {
                    $query->whereIn('branch_id', $branchIds);
                }
            })
            ->get();

        return Inertia::render('Chat', [
            'chats' => $chats,
            'calls' => $calls,
            'q' => $q,
            'branch_id' => $branchId,
            'branches' => $branches,
        ]);
    }

    public function show(Chat $chat)
    {
        $user = auth()->user();
        $isSuperAdmin = $user->hasRole('super_admin');

        if ($isSuperAdmin) {
            $branches = Branch::select('id', 'name')->get();
            $branchIds = $branches->pluck('id')->toArray();
        } else {
            $branches = collect();
            $branchIds = $user->branches()->pluck('id')->toArray();
        }

        // Get group chats for branches
        $groupChats = Chat::group()
            ->when(! $isSuperAdmin, function ($query) use ($branchIds) {
                $query->whereIn('branch_id', $branchIds);
            })
            ->with([
                'branch',
                'lastMessage',
                'lastMessage.media',
                'messages' => function ($query) use ($isSuperAdmin) {
                    if ($isSuperAdmin) {
                        $query->withTrashed();
                    }
                },
                'messages.media',
                'messages.sender',
                'messages.reactions',
                'messages.reactions.actor:id,name',
                'messages.replyTo' => function ($query) use ($isSuperAdmin) {
                    if ($isSuperAdmin) {
                        $query->withTrashed();
                    }
                },
                'messages.replyTo.sender',
                'messages.replyTo.media',
            ])
            ->withAggregate('lastMessage', 'created_at')
            ->withAggregate('lastMessage as last_message_sender_type', 'sender_type')
            ->orderByRaw('(last_message_sender_type = ?) DESC, last_message_created_at DESC', [User::class])
            ->get();

        // Get individual chats
        $individualChats = Chat::individual()
            ->when(! $isSuperAdmin, function ($query) use ($branchIds) {
                $query->whereHas('user', function (Builder $query) use ($branchIds) {
                    $query->whereIn('branch_id', $branchIds);
                });
            })
            ->withAggregate('lastMessage', 'created_at')
            ->withAggregate('lastMessage as last_message_sender_type', 'sender_type')
            ->orderByRaw('(last_message_sender_type = ?) DESC, last_message_created_at DESC', [User::class])
            ->with([
                'user',
                'user.dietPlan',
                'user.dietPlan.category',
                'user.lastVisit',
                'user.lastResult',
                'user.subscription',
                'lastMessage',
                'lastMessage.media',
                'messages' => function ($query) use ($isSuperAdmin) {
                    if ($isSuperAdmin) {
                        $query->withTrashed();
                    }
                },
                'messages.media',
                'messages.sender',
                'messages.reactions',
                'messages.reactions.actor:id,name',
                'messages.replyTo' => function ($query) use ($isSuperAdmin) {
                    if ($isSuperAdmin) {
                        $query->withTrashed();
                    }
                },
                'messages.replyTo.sender',
                'messages.replyTo.media',
            ])->get();

        // Merge group chats at the top, then individual chats
        $chats = $groupChats->concat($individualChats);

        $calls = SupportCall::with('user')
            ->where('status', SupportCallStatus::PENDING)
            ->whereHas('user', function (Builder $query) use ($branchIds) {
                $query->whereIn('branch_id', $branchIds);
            })
            ->get();

        return Inertia::render('Chat', [
            'chats' => $chats,
            'calls' => $calls,
            'selected_chat_id' => $chat->id,
            'branches' => $branches,
        ]);
    }

    public function store(Request $request, User $user, CreateChatAction $action)
    {
        $roomName = "support_chat_{$user->id}";

        $chat = $action->handle($user, $roomName);

        return redirect()->route('chat.show', $chat->id);
    }
}
