<?php

namespace App\Http\Controllers;

use App\Actions\CreateMessageAction;
use App\Enums\MessageType;
use App\Events\NewChatMessage;
use App\Http\Requests\ForwardMessageRequest;
use App\Http\Requests\StoreMessageRequest;
use App\Models\Admin;
use App\Models\Chat;
use App\Models\Message;
use Illuminate\Support\Facades\DB;

class MessageController extends Controller
{
    public function store(
        StoreMessageRequest $request,
        CreateMessageAction $action,
    )
    {
        /** @var Admin $user */
        $admin = auth()->user();

        $chat = Chat::query()->findOrFail(
            id: $request->integer('chat_id'),
        );
        if ($chat->admin_id !== $admin->id) {
            $chat->admin_id = $admin->id;
            $chat->save();
        }

        $message = $action->handle(
            chat: $chat,
            data: [
                'text' => $request->string('text')->toString(),
                'chat_id' => $request->integer('chat_id'),
                'reply_to_id' => $request->integer('reply_to_id') ?: null,
                'sender_id' => $admin->id,
                'sender_type' => $admin::class,
            ],
            attachments: $request->files->get("attachments"),
        );

        if ($request->has('tmp_front_id')) {
            $message->tmp_front_id = $request->string('tmp_front_id')->toString();
        }

        event(new NewChatMessage($chat, $message));

        return response()->json(
            data: [
                'message' => $message,
            ],
            status: 201,
        );
    }

    public function pin(Message $message)
    {
        $message->is_pinned = !$message->is_pinned;
        $message->save();

        return back();
    }

    /**
     * @throws \Throwable
     */
    public function forward(
        Message               $message,
        ForwardMessageRequest $request,
        CreateMessageAction   $action,
    )
    {
        /** @var Admin $admin */
        $admin = auth()->user();

        foreach ($request->chat_ids as $chatId) {
            DB::beginTransaction();

            $chat = Chat::findOrFail($chatId);

            $messageType = MessageType::TEXT;

            $newMessage = $chat->messages()->create([
                'type' => $messageType,
                'text' => $message->text,
                'chat_id' => $chatId,
                'sender_id' => $admin->id,
                'sender_type' => $admin::class,
            ]);

            if ($message->hasMedia('attachments')) {
                foreach ($message->getMedia('attachments') as $media) {
                    $media->copy($newMessage, 'attachments');
                }
            }

            $newMessage->load('media', 'replyTo', 'replyTo.sender');

            DB::commit();

            event(new NewChatMessage($chat, $newMessage));
        }

        return back();
    }

    public function destroy(Message $message)
    {
        $user = auth()->user();

        abort_unless($user->hasRole('super_admin'), 403);

        $message->delete();

        return back();
    }

    public function restore($id)
    {
        $user = auth()->user();

        abort_unless($user->hasRole('super_admin'), 403);

        $message = Message::withTrashed()->findOrFail($id);

        $message->restore();

        return back()->with('restored', true);
    }

}
