<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Casts\Attribute;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use Laravel\Sanctum\HasApiTokens;
use Spatie\Permission\Traits\HasRoles;

class Admin extends Authenticatable
{
    use HasApiTokens;
    use HasRoles;
    use Notifiable;

    protected string $guard = 'admin';

    protected $fillable = [
        'name',
        'email',
        'password',
        'avatar_url'
    ];

    protected $hidden = [
        'password', 'remember_token',
    ];

    protected $appends = [
        'photo_url',
    ];

    public function photoUrl(): Attribute
    {
        return Attribute::make(
            get: fn() => isset($this->avatar_url) ? Storage::url($this->avatar_url) : "https://ui-avatars.com/api/?name={$this->name}&amp;color=FFFFFF&amp;background=random",
        );
    }

    public function gender(): Attribute
    {
        return Attribute::make(
            get: fn() => null,
        );
    }

    public function branches(): BelongsToMany
    {
        return $this->belongsToMany(Branch::class);
    }

    public function ratefromable()
    {
        return $this->morphMany(ChatRate::class, 'ratefromable');
    }

    public function ratetoable()
    {
        return $this->morphMany(ChatRate::class, 'ratetoable');
    }

    public function chats()
    {
        return $this->hasMany(Chat::class);
    }


    public function getUnansweredMessagesCount(): int
    {
        $isSuper = $this->hasRole('super_admin');

        $base = DB::table('messages as m')
            ->where('m.sender_type', User::class)
            ->whereNotExists(function ($q) {
                $q->from('messages as a')
                    ->whereColumn('a.chat_id', 'm.chat_id')
                    ->where('a.sender_type', Admin::class)
                    ->whereRaw('(a.created_at, a.id) > (m.created_at, m.id)');
            });

        if ($isSuper) {
            return $base->count();
        }

        $branchIds = $this->branches()->pluck('branches.id');

        return $base
            ->join('chats as c', 'c.id', '=', 'm.chat_id')
            ->join('users as u', 'u.id', '=', 'c.user_id')
            ->whereIn('u.branch_id', $branchIds)
            ->count();
    }
}
