<?php

namespace App\Models;

// use Illuminate\Contracts\Auth\MustVerifyEmail;
use Illuminate\Database\Eloquent\Casts\Attribute;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\HasOne;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;
use Laravel\Sanctum\HasApiTokens;
use Spatie\MediaLibrary\HasMedia;
use Spatie\MediaLibrary\InteractsWithMedia;

class User extends Authenticatable implements HasMedia
{
    use HasApiTokens;

    /** @use HasFactory<\Database\Factories\UserFactory> */
    use HasFactory;

    use InteractsWithMedia;
    use Notifiable;

    /**
     * The attributes that are mass assignable.
     *
     * @var list<string>
     */
    protected $fillable = [
        'name',
        'email',
        'password',
    ];

    /**
     * The attributes that should be hidden for serialization.
     *
     * @var list<string>
     */
    protected $hidden = [
        'password',
        'remember_token',
    ];

    protected $appends = [
        'photo_url',
        'current_weight',
        'current_fats',
    ];

    /**
     * Get the attributes that should be cast.
     *
     * @return array<string, string>
     */
    protected function casts(): array
    {
        return [
            'email_verified_at' => 'datetime',
            'password' => 'hashed',
            'last_seen' => 'datetime',
        ];
    }

    public function chats(): HasMany
    {
        return $this->hasMany(Chat::class);
    }

    public function photoUrl(): Attribute
    {
        return Attribute::make(
            get: function () {
                $photo = $this->media()
                    ->where('collection_name', 'photo')
                    ->latest()
                    ->first();

                if ($photo) {
                    return $photo->getUrl();
                }

                $gender = $this->getAttribute('gender');

                if ($gender === 'male') {
                    return asset('images/default-male.png');
                }

                if ($gender === 'female') {
                    return asset('images/default-female.png');
                }

                return "https://ui-avatars.com/api/?name={$this->name}&color=FFFFFF&background=random";
            }
        );
    }

    public function subscriptions(): HasMany
    {
        return $this->hasMany(Subscription::class);
    }

    public function subscription(): HasOne
    {
        return $this->hasOne(Subscription::class)
            ->where('expires_on', '>', now())
            ->latestOfMany();
    }

    public function hasSubscription(): Attribute
    {
        return Attribute::make(
            get: fn () => $this->subscription()->exists(),
        );
    }

    public function branch(): BelongsTo
    {
        return $this->belongsTo(Branch::class);
    }

    public function DietPlan(): BelongsTo
    {
        return $this->belongsTo(DietPlan::class, 'diet_plan_id');
    }

    public function results(): HasMany
    {
        return $this->hasMany(UserWeekResult::class)
            ->orderBy('created_at', 'desc');
    }

    public function currentWeight(): Attribute
    {
        $this->loadMissing('lastResult');

        return new Attribute(
            get: fn () => $this->lastResult?->weight ?? $this->weight ?? 0
        );
    }

    public function lastResult()
    {
        return $this->hasOne(UserWeekResult::class)
            ->latestOfMany();
    }

    public function currentFats(): Attribute
    {
        return new Attribute(
            get: fn () => $this->lastResult?->fats ?? $this->fats ?? 0
        );
    }

    public function lastVisit()
    {
        return $this->hasOne(Visit::class)
            ->latestOfMany();
    }

    public function ratefromable()
    {
        return $this->morphMany(ChatRate::class, 'ratefromable');
    }

    public function ratetoable()
    {
        return $this->morphMany(ChatRate::class, 'ratetoable');
    }

    public function deviceTokens(): HasMany
    {
        return $this->hasMany(DeviceToken::class);
    }

    public function routeNotificationForFcm(): array
    {
        return $this->deviceTokens()->pluck('token')->toArray();
    }
}
