<?php

namespace App\Services;

use Agence104\LiveKit\AccessToken;
use Agence104\LiveKit\AccessTokenOptions;
use Agence104\LiveKit\BaseServiceClient;
use Agence104\LiveKit\RoomCreateOptions;
use Agence104\LiveKit\RoomServiceClient;
use Agence104\LiveKit\VideoGrant;
use Exception;
use Livekit\Room;

final readonly class LiveKitService
{
    private BaseServiceClient $roomClient;
    protected string $apiKey;
    protected string $apiSecret;

    /**
     * @throws Exception
     */
    public function __construct()
    {
        $this->apiKey = config('livekit.api_key');
        $this->apiSecret = config('livekit.api_secret');

        $this->roomClient = new RoomServiceClient(
            host: config('livekit.url'),
            apiKey: $this->apiKey,
            apiSecret: $this->apiSecret,
        );
    }

    /**
     * Create a new LiveKit room.
     *
     * @param string $roomName
     * @param ?int $maxParticipants (optional)
     * @return array{success: bool, room?: Room, error?: ?string}
     */
    public function createRoom(string $roomName, ?int $maxParticipants = null): array
    {
        try {
            $opts = (new RoomCreateOptions())
                ->setName($roomName)
                ->setEmptyTimeout(config('livekit.empty_room_timeout_seconds'));

            if ($maxParticipants) {
                $opts = $opts->setMaxParticipants($maxParticipants);
            }

            $room = $this->roomClient->createRoom($opts);

            return [
                'success' => true,
                'room' => $room,
            ];
        } catch (Exception $e) {
            return [
                'success' => false,
                'error' => $e->getMessage(),
            ];
        }
    }

    /**
     * Generate an access token for a participant.
     *
     * @param string $roomName
     * @param string $participantIdentity
     * @param bool $canPublish
     * @param bool $canSubscribe
     * @return array{success: bool, token?: string, error?: string}
     */
    public function generateAccessToken(
        string $roomName,
        string $participantIdentity,
        bool   $canPublish = true,
        bool   $canSubscribe = true
    ): array
    {
        try {
            $tokenOptions = (new AccessTokenOptions())
                ->setIdentity($participantIdentity);
            $videoGrant = new VideoGrant();
            $videoGrant->setRoomJoin()
                ->setRoomName($roomName)
                ->setCanPublish($canPublish)
                ->setCanSubscribe($canSubscribe);

            $token = (new AccessToken($this->apiKey, $this->apiSecret))
                ->init($tokenOptions)
                ->setGrant($videoGrant)
                ->toJwt();

            return [
                'success' => true,
                'token' => $token,
            ];
        } catch (Exception $e) {
            return [
                'success' => false,
                'error' => $e->getMessage(),
            ];
        }
    }
}
