/**
 * File utility functions for handling file operations in the chat application
 */

/**
 * Converts file size in bytes to human readable format
 * @param {number} bytes - File size in bytes
 * @param {boolean} si - Use SI units (1000) instead of binary (1024)
 * @param {number} dp - Decimal places to show
 * @returns {string} Human readable file size
 */
export function humanFileSize(bytes, si = false, dp = 1) {
    if (!bytes && bytes !== 0) return '';
    if (bytes < 1024) return `${bytes} B`;

    const thresh = si ? 1000 : 1024;
    const units = si
        ? ['kB', 'MB', 'GB', 'TB', 'PB', 'EB', 'ZB', 'YB']
        : ['KB', 'MB', 'GB', 'TB', 'PB', 'EB', 'ZB', 'YB'];

    let u = 0;
    while (bytes >= thresh && u < units.length - 1) {
        bytes /= thresh;
        u++;
    }

    return `${bytes.toFixed(dp)} ${units[u]}`;
}

/**
 * Converts MIME type to human readable file type
 * @param {string} mimeType - The MIME type string
 * @returns {string} Human readable file type
 */
export function humanReadableFileType(mimeType) {
    const typeMap = {
        'image/jpeg': 'JPEG Image',
        'image/png': 'PNG Image',
        'image/gif': 'GIF Image',
        'image/webp': 'WebP Image',
        'video/mp4': 'MP4 Video',
        'video/quicktime': 'QuickTime Video',
        'audio/mpeg': 'MP3 Audio',
        'audio/wav': 'WAV Audio',
        'application/pdf': 'PDF Document',
        'application/msword': 'Word Document',
        'application/vnd.openxmlformats-officedocument.wordprocessingml.document':
            'Word Document',
        'application/vnd.ms-excel': 'Excel Spreadsheet',
        'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet':
            'Excel Spreadsheet',
        'application/vnd.ms-powerpoint': 'PowerPoint Presentation',
        'application/vnd.openxmlformats-officedocument.presentationml.presentation':
            'PowerPoint Presentation',
        'text/plain': 'Text File',
        'text/csv': 'CSV File',
        'application/zip': 'ZIP Archive',
        'application/vnd.rar': 'RAR Archive',
        'application/vnd.android.package-archive': 'APK File',
    };

    return (
        typeMap[mimeType] ||
        mimeType?.split('/')[1]?.toUpperCase() + ' File' ||
        'File'
    );
}

/**
 * Categorizes files by type (images, videos, audios, other)
 * @param {Array} media - Array of media files
 * @returns {Object} Categorized files object
 */
export function categorizeFiles(media) {
    return (media || []).reduce(
        (acc, file) => {
            if (file.mime_type?.startsWith('image/')) {
                acc.images.push(file);
            } else if (file.mime_type?.startsWith('video/')) {
                acc.videos.push(file);
            } else if (file.mime_type?.startsWith('audio/')) {
                acc.audios.push(file);
            } else {
                acc.otherFiles.push(file);
            }
            return acc;
        },
        { images: [], videos: [], audios: [], otherFiles: [] },
    );
}
